// Buffer.cpp: implementation of the CBuffer class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Buffer.h"
#include "NsLock.h"
#include <Math.h>

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

 
namespace nsComm
{

	//////////////////////////////////////////////////////////////////////
	// Construction/Destruction
	//////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	CBuffer
	// 
	// DESCRIPTION:	Constructs the buffer with a default size
	// 
	// RETURNS:		
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	CBuffer::CBuffer()
	{
		// Initial size
		m_nSize = 0;
		InitializeCriticalSection(&m_BufferCriti);
		m_pPtr = m_pBase = NULL;
	}

	CBuffer::CBuffer(CBuffer* pOther)
	{
		//if (this != pOther)
		//{
		//	*this = *pOther;
		//}
		m_nSize = 0;
		InitializeCriticalSection(&m_BufferCriti);
		m_pPtr = m_pBase = NULL;
		Copy(*pOther);
	}

	CBuffer& CBuffer::operator=(const CBuffer& other)
	{
		if (this != &other)
		{
			m_pBase	=	other.m_pBase;
			m_pPtr	=	other.m_pPtr;
			m_nSize	=	other.m_nSize;
		}
		return *this;
	}
	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	~CBuffer
	// 
	// DESCRIPTION:	Deallocates the buffer
	// 
	// RETURNS:		
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	CBuffer::~CBuffer()
	{
		DeleteCriticalSection(&m_BufferCriti);
		if (m_pBase)
			VirtualFree(m_pBase,0,MEM_RELEASE);
	}
		

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	Write
	// 
	// DESCRIPTION:	Writes data into the buffer
	// 
	// RETURNS:		
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	BOOL CBuffer::Write(PBYTE pData, UINT nSize)
	{
		nsHelper::CNsLock singleLock(&m_BufferCriti);
		singleLock.Lock();

		ReAllocateBuffer(nSize + GetBufferLen());

		CopyMemory(m_pPtr,pData,nSize);

		// Advance Pointer
		m_pPtr+=nSize;

		singleLock.Unlock();

		return nSize;
	}

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	Insert
	// 
	// DESCRIPTION:	Insert data into the buffer 
	// 
	// RETURNS:		
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	BOOL CBuffer::Insert(PBYTE pData, UINT nSize)
	{
		nsHelper::CNsLock singleLock(&m_BufferCriti);
		singleLock.Lock();

		ReAllocateBuffer(nSize + GetBufferLen());

		MoveMemory(m_pBase+nSize,m_pBase,GetMemSize() - nSize);
		CopyMemory(m_pBase,pData,nSize);

		// Advance Pointer
		m_pPtr+=nSize;

		singleLock.Unlock();

		return nSize;
	}


	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	Read
	// 
	// DESCRIPTION:	Reads data from the buffer and deletes what it reads
	// 
	// RETURNS:		
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	UINT CBuffer::Read(PBYTE pData, UINT nSize, BOOL bDelAfterRead)
	{
		// Trying to byte off more than ya can chew - eh?
		if (nSize > GetMemSize())
			return 0;

		nsHelper::CNsLock singleLock(&m_BufferCriti);
		singleLock.Lock();


		// all that we have 
		if (nSize > GetBufferLen())
			nSize = GetBufferLen();

			
		if (nSize)
		{
			// Copy over required amount and its not up to us
			// to terminate the buffer - got that!!!
			CopyMemory(pData,m_pBase,nSize);
			
			if (bDelAfterRead)
			{
				// Slide the buffer back - like sinking the data
				MoveMemory(m_pBase,m_pBase+nSize,GetMemSize() - nSize);
				m_pPtr -= nSize;
			}
		}
			
		DeAllocateBuffer(GetBufferLen());

		singleLock.Unlock();

		return nSize;
	}

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	GetMemSize
	// 
	// DESCRIPTION:	Returns the phyical memory allocated to the buffer
	// 
	// RETURNS:		
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	UINT CBuffer::GetMemSize() 
	{
		return m_nSize;
	}

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	GetBufferLen
	// 
	// DESCRIPTION:	Get the buffer 'data' length
	// 
	// RETURNS:		
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	UINT CBuffer::GetBufferLen() 
	{
		if (m_pBase == NULL)
			return 0;

		int nSize = 
			m_pPtr - m_pBase;
		return nSize;
	}

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	ReAllocateBuffer
	// 
	// DESCRIPTION:	ReAllocateBuffer the Buffer to the requested size
	// 
	// RETURNS:		
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	UINT CBuffer::ReAllocateBuffer(UINT nRequestedSize)
	{
		if (nRequestedSize < GetMemSize())
			return 0;

		// Allocate new size
		UINT nNewSize = (UINT) ceil(nRequestedSize / 1024.0) * 1024;

		// New Copy Data Over
		PBYTE pNewBuffer = (PBYTE) VirtualAlloc(NULL,nNewSize,MEM_COMMIT,PAGE_READWRITE);

		UINT nBufferLen = GetBufferLen();
		CopyMemory(pNewBuffer,m_pBase,nBufferLen);

		if (m_pBase)
			VirtualFree(m_pBase,0,MEM_RELEASE);


		// Hand over the pointer
		m_pBase = pNewBuffer;

		// Realign position pointer
		m_pPtr = m_pBase + nBufferLen;

		m_nSize = nNewSize;

		return m_nSize;
	}

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	DeAllocateBuffer
	// 
	// DESCRIPTION:	DeAllocates the Buffer to the requested size
	// 
	// RETURNS:		
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	UINT CBuffer::DeAllocateBuffer(UINT nRequestedSize)
	{
		if (nRequestedSize < GetBufferLen())
			return 0;

		// Allocate new size
		UINT nNewSize = (UINT) ceil(nRequestedSize / 1024.0) * 1024;

		if (nNewSize < GetMemSize())
			return 0;

		// New Copy Data Over
		PBYTE pNewBuffer = (PBYTE) VirtualAlloc(NULL,nNewSize,MEM_COMMIT,PAGE_READWRITE);

		UINT nBufferLen = GetBufferLen();
		CopyMemory(pNewBuffer,m_pBase,nBufferLen);

		VirtualFree(m_pBase,0,MEM_RELEASE);

		// Hand over the pointer
		m_pBase = pNewBuffer;

		// Realign position pointer
		m_pPtr = m_pBase + nBufferLen;

		m_nSize = nNewSize;

		return m_nSize;
	}

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	Scan
	// 
	// DESCRIPTION:	Scans the buffer for a given byte sequence
	// 
	// RETURNS:		Logical offset
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	int CBuffer::Scan(PBYTE pScan,UINT nPos)
	{
		if (nPos > GetBufferLen() )
			return -1;

		PBYTE pStr = (PBYTE) strstr((char*)(m_pBase+nPos),(char*)pScan);
		
		int nOffset = 0;

		if (pStr)
			nOffset = (pStr - m_pBase) + strlen((char*)pScan);

		return nOffset;
	}

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	ClearBuffer
	// 
	// DESCRIPTION:	Clears/Resets the buffer
	// 
	// RETURNS:	
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	void CBuffer::ClearBuffer()
	{
		// Force the buffer to be empty
		m_pPtr = m_pBase;

		DeAllocateBuffer(1024);
	}

 

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	Copy
	// 
	// DESCRIPTION:	Copy from one buffer object to another...
	// 
	// RETURNS:	
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	void CBuffer::Copy(CBuffer& buffer)
	{
		int nReSize = buffer.GetMemSize();
		int nSize = buffer.GetBufferLen();
		ClearBuffer();
		ReAllocateBuffer(nReSize);

		m_pPtr = m_pBase + nSize;

		CopyMemory(m_pBase,buffer.GetBuffer(),buffer.GetBufferLen());
	}

	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	GetBuffer
	// 
	// DESCRIPTION:	Returns a pointer to the physical memory determined by the offset
	// 
	// RETURNS:	
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	PBYTE CBuffer::GetBuffer(UINT nPos)
	{
		return m_pBase+nPos;
	}



	////////////////////////////////////////////////////////////////////////////////
	// 
	// FUNCTION:	Delete
	// 
	// DESCRIPTION:	Delete data from the buffer and deletes what it reads
	// 
	// RETURNS:		
	// 
	// NOTES:	
	// 
	// MODIFICATIONS:
	// 
	// Name				Date		Version		Comments
	// N T ALMOND       270400		1.0			Origin
	// 
	////////////////////////////////////////////////////////////////////////////////
	UINT CBuffer::Delete(UINT nSize)
	{
		// Trying to byte off more than ya can chew - eh?
		if (nSize > GetMemSize())
			return 0;

		// all that we have 
		if (nSize > GetBufferLen())
			nSize = GetBufferLen();

			
		if (nSize)
		{
			// Slide the buffer back - like sinking the data
			MoveMemory(m_pBase,m_pBase+nSize,GetMemSize() - nSize);

			m_pPtr -= nSize;
		}
			
		DeAllocateBuffer(GetBufferLen());

		return nSize;
	}
}
